/*************************************************
*     rpld - an IBM style RIPL server            *
*************************************************/

/* Copyright (c) 1999,2000, James McKenzie.
 *                      All rights reserved
 * Copyright (c) 1998,2000, Christopher Lightfoot.
 *                      All rights reserved
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENCE file which can be found at the top level of
 * the rpld distribution.
 *
 * IBM is a trademark of IBM corp.
 *
 */

/*
 *	YACC grammar for RPLD conf file parser
 *
 * $Log: rpld_conf.y,v $
 * Revision 1.7  2000/07/23 19:14:19  root
 * #
 *
 * Revision 1.6  2000/07/17 11:59:45  root
 * #
 *
 * Revision 1.5  2000/07/17 10:43:34  root
 * #
 *
 * Revision 1.4  2000/07/16 14:05:28  root
 * #
 *
 * Revision 1.3  2000/07/16 13:18:10  root
 * #
 *
 * Revision 1.1  2000/07/16 13:16:33  root
 * #
 *
 * Revision 1.2  1999/09/13 11:17:35  root
 * \#
 *
 * Revision 1.1  1999/09/13 11:04:13  root
 * \#
 *
 * Revision 1.10  1999/09/12 19:45:03  chris
 * *** empty log message ***
 *
 * Revision 1.9  1999/09/12 19:14:07  chris
 * Error messages now report name of last token scanned, instead of current yytext.
 *
 * Revision 1.8  1999/09/12 17:39:01  chris
 * Configuration file now correctly builds structures; various minor problems fixed.
 *
 * Revision 1.7  1999/09/12 04:21:29  chris
 * Wrote back-end to parser.
 *
 * Revision 1.6  1999/09/12 03:27:35  chris
 * Added better error reporting.
 *
 * Revision 1.5  1999/09/12 01:05:00  chris
 * Supports detecting start and end of blocks.
 *
 * Revision 1.4  1999/09/12 00:58:02  chris
 * Added named block syntax.
 *
 * Revision 1.3  1999/09/11 19:00:51  chris
 * Added support for nested blocks.
 *
 * Revision 1.2  1999/09/11 18:53:41  chris
 * Added a comment to say what the file does.
 *
 *
 */

%{

static char rcsid[]="$Id: rpld_conf.y,v 1.7 2000/07/23 19:14:19 root Exp root $";

#include "project.h"

// state machine stuff

typedef enum {START, BLOCK_START, BLOCK_END, ASSERTION, ASSIGNMENT} THING ;
typedef enum {INIT, GLOBALBLOCK, HOSTBLOCK, FILEBLOCK} STATE ;

//void process_thing(THING thing, char *name, int type, YYSTYPE *pvalue);

%}

%token BLOCK_START BLOCK_END NAME TEXT NUMBER MACADDR MACADDR_PARTIAL

%union {
		long number;
		char *name;
		char *text;
		char mac_address[6];
		struct partial_mac {
			char mac_address[6];
			int mac_len;
		} pm;
	}

%start block_list

%%

block_list:	block ';'
	|	block_list block ';'
	;

block:		block_start statement_list BLOCK_END	{ process_thing(BLOCK_END, "", 0, NULL); }
	|	block_start BLOCK_END	{ process_thing(BLOCK_END, "", 0, NULL); }

block_start:	NAME BLOCK_START	{ process_thing(BLOCK_START, $1.name, 0, NULL); }
	|	BLOCK_START		{ process_thing(BLOCK_START, "", 0, NULL); }

statement_list:	';'
	|	statement ';'
	|	statement_list statement ';'


statement:	NAME			{ process_thing(ASSERTION, $1.name, 0, NULL); }
	|	NAME '=' TEXT 		{ process_thing(ASSIGNMENT, $1.name, TEXT, &$3); }
	|	NAME '=' NUMBER		{ process_thing(ASSIGNMENT, $1.name, NUMBER, &$3); }
	|	NAME '=' MACADDR	{ process_thing(ASSIGNMENT, $1.name, MACADDR, &$3); }
	|	NAME '=' MACADDR_PARTIAL	{ process_thing(ASSIGNMENT, $1.name, MACADDR_PARTIAL, &$3); }
	|	block

%%

//
// ERROR REPORTING
//

// the lineno variable from our parser
extern int lineno;

// the yytext variable from lex for error reporting
extern char* yytext;

void yyerror(char *s)
{
	fprintf(stderr, "rpld: config line %d: %s near `%s'\n", lineno, s, yytext);
}

//
// CONFIGURATION PROCESSOR
//

// This is the bit that actually does the work

#define strsame(a, b)	(!strcmp((a), (b)))
/*
struct global_parameters
{

} g_params;
*/

struct clientinfo
{
	int have_mac;
	int have_run_addr;
	int have_files;
	// optional
	int have_framesize;
	int have_blocksize;
	int have_pacing;
	int have_nospew;
};

struct clfileinfo
{
	int have_path;
	int have_load_addr;
	// optional
	int have_offset;
	int have_length;
};

extern struct client *clients;

// hideous, we need to cope with a semicolon after this
//#define THROW_ERROR(a)	do { yyerror((a)); exit(1); } while(0)
#define THROW_ERROR(a)	do { fprintf(stderr, "rpld: config line %d: %s near `%s'\n", lineno, (a), name); exit(1); } while(0)

void process_thing(THING thing, char *name, int type, YYSTYPE *pvalue)
{
	static STATE state;
	static struct client *pc;
	static struct clientinfo ci;
	static struct clfile *pcf;
	static struct clfileinfo cfi;

	switch(thing)
	{
	// boot the state machine
	case START:
		state = INIT;
		break;

	//
	// Blocks, which contain related options
	//
	
	// start of a block
	case BLOCK_START:
		// in initial state, move to GLOBALBLOCK or HOSTBLOCK
		if (state == INIT) {
			if (strsame(name, "GLOBAL")) {
				state = GLOBALBLOCK;
				break;
			} else if (strsame(name, "HOST") || strsame(name, "CLIENT")) {
				// construct a new client entity
				pc = (struct client*)malloc(sizeof(struct client));
				bzero(pc, sizeof(struct client));
				// reset info about what options have been set for this client
				bzero(&ci, sizeof(ci));

				pc->blocklen=MY_BLOCK_LEN;
				pc->framelen=MY_FRAME_LEN;
				pc->pacing=MY_PACING;
				
				state = HOSTBLOCK;
				break;
			} else THROW_ERROR("Unknown top-level parameter block");
		}
		// in a HOST block, this must be a FILE
		else if (state == HOSTBLOCK) {
			if (strsame(name, "FILE")) {
				// construct a new file entity
				pcf = (struct clfile*)malloc(sizeof(struct clfile));
				bzero(pcf, sizeof(struct clfile));
				pcf->length=-1;
				// reset info about options set for this file
				bzero(&cfi, sizeof(cfi));
				
				state = FILEBLOCK;
				break;
			} else THROW_ERROR("Only a FILE parameter block can be included in a HOST block");
		}
		// fuck knows
		else
		{
			yyerror("Unknown parameter block");
			exit(1);
		}
		break;

	// end of a block, we should have a bunch of info now
	case BLOCK_END:
		// end GLOBAL block
		if (state == GLOBALBLOCK) {
			// no more global params, at least for the moment
		}
		// end HOST block
		else if (state == HOSTBLOCK) {
			// should have a complete host specification
			if (!ci.have_mac) THROW_ERROR("Must specify an ethernet (MAC) address for host");
			else if (!ci.have_run_addr) THROW_ERROR("Must specify an initial execute address for host");
			else if (!ci.have_files) THROW_ERROR("Must specify at least one file to load for host");

			// OK, should have an entire host spec, so copy it in
			pc->next = clients;
			clients = pc;

			// finished this host spec
			state = INIT;
			break;
		}
		// end FILE block
		else if (state == FILEBLOCK) {
			// should have a complete file specification
			if (!cfi.have_path) THROW_ERROR("Must specify a path for file");
			else if (!cfi.have_load_addr) THROW_ERROR("Must specify a load address for file");

			// have an entire file spec, copy it into the host spec
			pcf->next = pc->files;
			pc->files = pcf;

			ci.have_files = 1;

			// done
			state = HOSTBLOCK;
			break;
		}

	//
	// The various things that go inside blocks
	//
	
	case ASSERTION:
		if (state == GLOBALBLOCK) {
			// no global assertions ATM
			THROW_ERROR("Unknown directive");
		} else if (state == HOSTBLOCK) {
			// no host assertions ATM
			if (strsame(name,"nospew")) {
				if (ci.have_pacing) THROW_ERROR("Directive nospew incompatible with pacing ");
				if (ci.have_nospew) THROW_ERROR("Repeated directive");
				ci.have_nospew=1;
				pc->nospew++;
			} else THROW_ERROR("Unknown directive");
		} else if (state == FILEBLOCK) {
			if (strsame(name,"linux")) {
				if (!cfi.have_path) THROW_ERROR("A path to a
valid kernel must precede linux");

				do_linux_kernel(pc,pcf);
				cfi.have_load_addr=1;
				cfi.have_offset=1;
				cfi.have_length=1;
				ci.have_run_addr=1;
			}else{
			THROW_ERROR("Unknown directive");
			}
		} else THROW_ERROR("Unknown directive");
		break;

	case ASSIGNMENT:
		if (state == GLOBALBLOCK) {
			// no global assignments ATM
			THROW_ERROR("Unknown directive");
		} else if (state == HOSTBLOCK) {
			// ethernet address
			if (strsame(name, "ethernet") || strsame(name, "mac")) {
				if (type != MACADDR && type != MACADDR_PARTIAL)
					THROW_ERROR("Directive must be followed by a (partial or complete) ethernet address");
				else if (ci.have_mac) THROW_ERROR("Repeated directive");

				if (type == MACADDR) {
					// set MAC address; this is non-partial, so len = ETH_ALEN
					bcopy(pvalue->mac_address, pc->mac, ETH_ALEN);
					pc->partial_mac_len = ETH_ALEN;
				} else {
					bcopy(pvalue->pm.mac_address, pc->mac, pvalue->pm.mac_len);
					pc->partial_mac_len = pvalue->pm.mac_len;
				}

				ci.have_mac = 1;
			}
			// execute address
			else if (strsame(name, "execute") || strsame(name, "run")) {
				if (type != NUMBER) THROW_ERROR("Directive must be followed by a memory address");
				else if (ci.have_run_addr==2) THROW_ERROR("Repeated directive");

				// set address
				pc->run_addr = pvalue->number;
				ci.have_run_addr = 2;
			}
			else if (strsame(name, "blocksize")) {
				if (type != NUMBER) THROW_ERROR("Directive must be followed by an integer ");
				else if (ci.have_blocksize) THROW_ERROR("Repeated directive");

				// set address
				pc->blocklen = pvalue->number;
				ci.have_blocksize = 1;
			}
			else if (strsame(name, "framesize")) {
				if (type != NUMBER) THROW_ERROR("Directive must be followed by an integer ");
				else if (ci.have_framesize) THROW_ERROR("Repeated directive");

				// set size
				pc->framelen = pvalue->number;
				ci.have_framesize = 1;
			}
			else if (strsame(name,"pacing")) {
				if (type != NUMBER) THROW_ERROR("Directive must be followed by an integer ");
				if (ci.have_nospew) THROW_ERROR("Directive pacing incompatible with nospew ");
				if (ci.have_pacing) THROW_ERROR("Repeated directive");
				pc->pacing = pvalue->number;
				ci.have_pacing =1;
			}
			else THROW_ERROR("Unknown directive");
		} else if (state == FILEBLOCK) {
			// path
			if (strsame(name, "path")) {
				if (type != TEXT) THROW_ERROR("Directive must be followed by a filename");
				else if (cfi.have_path) THROW_ERROR("Repeated directive");

				// set filename
				pcf->path = pvalue->text;
				cfi.have_path = 1;
			}
			// load address
			else if (strsame(name, "load")) {
				if (type != NUMBER) THROW_ERROR("Directive must be followed by a memory address");
				else if (cfi.have_load_addr) THROW_ERROR("Repeated directive");

				// set load address
				pcf->load_addr = pvalue->number;
				cfi.have_load_addr = 1;
			}
			// offset
			else if (strsame(name, "offset")) {
				if (type != NUMBER) THROW_ERROR("Directive must be followed by a memory address");
				else if (cfi.have_offset) THROW_ERROR("Repeated directive");

				// set offset
				pcf->offset = pvalue->number;
				cfi.have_offset = 1;
			}
			// length
			else if (strsame(name, "length")) {
				if (type != NUMBER) THROW_ERROR("Directive must be followed by a memory address");
				else if (cfi.have_length) THROW_ERROR("Repeated directive");

				// set length
				pcf->length = pvalue->number;
				cfi.have_length = 1;
			}
			else THROW_ERROR("Unknown directive");
		}
		break;

		default: THROW_ERROR("Mistake");
	}
	

}
